package Esfinge::Filtros;

use strict;
use warnings;


our $VERSION = '0.01';

# Dependncias

use jspell;
use Lingua::PT::PLNbase; 


=head1 FUNCTIONS

=head2 new

Construtor

=cut


sub new
{
  my $this = {};
  $this->{objectoJspell} = jspell::new("port");
  $this->{PoSInteressantes} = (); # Lista de categorias gramaticais consideradas interessantes para obter respostas (array)
  $this->{RespostasIndesejaveis} = (); # Lista de respostas indesejveis. (array)
  bless $this;
  return $this;
}


sub FazParteExpressao ($ $)

# Esta funo verifica se as palavras que constituem uma resposta esto incluidas numa expresso (apenas so consideradas as palavras que podem pertencer a uma das categorias gramaticais consideradas interessantes para obter respostas).
# ENTRADA 1: Resposta. (string)
# ENTRADA 2: Expresso (pode ser por exemplo uma pergunta, ou um padro usado para fazer uma pesquisa). (string)
# SADA: String contendo apenas as palavras que podem pertencer a uma das categorias gramaticais consideradas interessantes para obter respostas. (string)

{

  my $this = shift;
  my $Resposta = shift;
  my $Expressao = shift;

  my $fazParte = 0;

  $Resposta = $this->PalavrasPoSInteressante ($Resposta);  

  $Expressao = $this->PalavrasPoSInteressante ($Expressao);

  my @atomos = tokeniza ($Resposta);

  foreach my $atomo (@atomos) {

    if ($Expressao =~ m/.*$atomo.*/i) {

      $fazParte = 1;

    }
  }

  return $fazParte;
}


sub RespostaIndesejavel ($)

# Esta funo verifica se uma resposta est includa numa lista de respostas indesejveis
# ENTRADA 1: Uma resposta. (string)
# SADA: 1 caso a resposta esteja na lista de respostas indesejveis, 0 no caso contrrio. (inteiro)


{ my $this = shift;
  my $Resposta = shift;
  
  if (grep {$Resposta=~ m/^$_$/gi} @{$this->{RespostasIndesejaveis}}) {
    return 1; 
  }
  else {
    return 0;
  } 

}




sub PalavrasPoSInteressante ($)

# Esta funo recebe uma string com palavras e devolve outra string contendo apenas as palavras que podem pertencer a uma das categorias gramaticais consideradas interessantes para obter respostas.
# ENTRADA 1: Palavras. (string)
# SADA: String contendo apenas as palavras que podem pertencer a uma das categorias gramaticais consideradas interessantes para obter respostas. (string)

{

  my $this = shift;
  my $Palavras = shift;
  my $PalavrasInteressantes = "";  

  my @atomos = tokeniza ($Palavras);

  foreach my $atomo (@atomos) {

    if ($this->PoSInteressante ($atomo)) {

      $PalavrasInteressantes .= $atomo . " ";

    }

  }

  return $PalavrasInteressantes;

}




sub PoSInteressante ($)

# Esta funo utiliza o jspell para verificar se uma palavra pode pertencer a uma das categorias gramaticais consideradas interessantes para obter respostas.
# ENTRADA 1: Uma palavra. (string)
# SADA: 1 caso a palavra possa pertencer a uma das categorias gramticais consideradas interessantes para obter respostas, 0 no caso contrrio. (inteiro)

{
  my $this = shift;

  my $PalavraAnalisada = shift;

  my @categorias = $this->{objectoJspell}->fea($PalavraAnalisada);

  my $interessante = 0;

  if (!($categorias [0]->{CAT})) {

    $interessante = 1; # O jspell no consegue classificar.

  }
  else {
    for (my $i=0;$i!=$#{@categorias} + 1;$i++) {

      if (grep {$categorias [$i]->{CAT} eq $_} @{$this->{PoSInteressantes}}) {
      
        $interessante = 1;

      }
    }
  }
  return $interessante;

}


sub PoSInteressantes ($)

# Esta funo verifica se as palavras que constituem uma resposta pertencem a categorias gramaticais interessantes. Para isso verifica se a primeira e ltima palavras 
# podem pertencer a uma das categorias gramaticais consideradas interessantes.
# ENTRADA 1: Uma resposta. (string)
# SADA: 1 caso as palavras da resposta tenham categorias gramaticais interessantes, 0 no caso contrrio. (inteiro)


{
  my $this = shift;
  my $Resposta = shift;
    
  $Resposta =  tokeniza($Resposta);
  my @atomos = split /\s/, $Resposta; # produz um array de palavras

  if ($this->PoSInteressante ($atomos[0]) and
      $this->PoSInteressante ($atomos [$#{@atomos}])) {
    return 1; 
  }
  else {
    return 0;
  } 

 
}

sub DESTROY
{
        my ($self) = @_;
}


1;


__END__

=head1 NAME

Esfinge::Filtros

=head1 SYNOPSIS


Este mdulo contm mtodos que podem ser usados para filtrar respostas. Inclui um mtodo que indica se uma resposta inclui palavras contidas numa expresso (apenas so consideradas as palavras com o PoS interessante que foram definidas no construtor desta classe), um mtodo para verificar se uma resposta est includa numa lista pre-definida de respostas indesejveis, um mtodo que dada uma string com palavras retorna apenas aquelas que tiverem PoS interessante e finalmente um mtodo que indica se uma resposta tem as primeira e ltima palavras com PoS interessante.

#Exemplo de utilizao da package Filtros

#!/usr/local/bin/perl

use Filtros;

# Resposta a analisar.

$Resposta= "Amor  fogo";

# Expresso a analisar

$Expressao= "Amor  fogo que arde sem se ver";

# Ficheiro com respostas indesejveis.

$NomeFicheiro="respostas-indesejaveis.txt";

$ObjectoFiltro = Filtros::new();

# Carregamento das categorias gramaticais consideradas interessantes para obter respostas para a varivel RespostasIndesejaveis do objecto $ObjectoFiltro.

@{$ObjectoFiltro->{PoSInteressantes}} = ("adj", "nc", "card", "a_nc", "np"); 


if ($ObjectoFiltro->PoSInteressantes($Resposta)) {

  print "\n\"$Resposta\"  uma resposta interessante\n";

}
else {

  print "\n\"$Resposta\" no  uma resposta interessante\n";

}


# Carregamento das respostas indesejveis para a varivel RespostasIndesejaveis do objecto $ObjectoFiltro
  
open (INDESEJ, "$NomeFicheiro");
  
while (my $palavra = <INDESEJ>) {

  $palavra =~ s/\n//;
  push  @{$ObjectoFiltro->{RespostasIndesejaveis}}, $palavra;

}


if ($ObjectoFiltro->RespostaIndesejavel($Resposta)) {

  print "\n\"$Resposta\"  uma resposta indesejvel\n";

}
else {

  print "\n\"$Resposta\" no  uma resposta indesejvel\n";

}


print "\n Palavras com PoS interessante: " . $ObjectoFiltro->PalavrasPoSInteressante ($Resposta) . "\n";


if ($ObjectoFiltro->FazParteExpressao ($Resposta, $Expressao)) {

    print "\n\"$Resposta\" faz parte da expresso \"$Expressao\"\n";

}
else {

    print "\n\"$Resposta\" no faz parte da expresso \"$Expressao\"\n";

}


=head1 SEE ALSO

=head1 AUTHOR

Lus Fernando Costa, E<lt>luis.f.kosta@gmail.comE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2008 by Linguateca

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.8.6 or,
at your option, any later version of Perl 5 you may have available.


=cut
